/*
 * Copyright (c) 1999-2001 Lutris Technologies, Inc. All Rights
 * Reserved.
 * 
 * This source code file is distributed by Lutris Technologies, Inc. for
 * use only by licensed users of product(s) that include this source
 * file. Use of this source file or the software that uses it is covered
 * by the terms and conditions of the Lutris Enhydra Development License
 * Agreement included with this product.
 * 
 * This Software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
 * ANY KIND, either express or implied. See the License for the specific terms
 * governing rights and limitations under the License.
 * 
 * Contributor(s):
 * 
 * $Id: WbxmlWriter.java,v 1.1.1.1 2001/01/05 05:11:17 sese0235 Exp $
 */

package de.kxml.wap;

import java.io.*;
import java.util.*;
import de.kxml.*;
import de.kxml.io.*;

// import com.sun.xml.parser.Parser;

/**
 * a class for converting ("binary encoding") XML to WBXML.
 * Todo:
 * <ul>
 * <li>Add support for processing instructions
 * <li>Add support for tag and attribute tables
 * <li>Add support for WBXML extensions
 * </ul>
 */
public class WbxmlWriter extends AbstractXmlWriter {
    Hashtable		  stringTable = new Hashtable();
    OutputStream	  out;
    ByteArrayOutputStream buf = new ByteArrayOutputStream();
    ByteArrayOutputStream stringTableBuf = new ByteArrayOutputStream();
    String		  pending;
    Vector		  attributes = new Vector();

    /**
     * Constructor declaration
     *
     *
     * @param out
     *
     * @see
     */
    public WbxmlWriter(OutputStream out) throws IOException {
	this.out = out;
	buf = new ByteArrayOutputStream();
	stringTableBuf = new ByteArrayOutputStream();

	// ok, write header
	out.write(0x01);    // version
	out.write(0x01);    // unknown or missing public identifier
	out.write(0x04);    // iso-8859-1
    }

    /**
     * ATTENTION: flush cannot work since Wbxml documents cannot
     * need buffering. Thus, this call does nothing.
     */
    public void flush() {}

    /**
     * Method declaration
     *
     *
     * @throws IOException
     *
     * @see
     */
    public void close() throws IOException {
	writeInt(out, stringTableBuf.size());

	// write StringTable
	out.write(stringTableBuf.toByteArray());

	// write buf
	out.write(buf.toByteArray());

	// ready!
	out.flush();
    } 

    /**
     * Method declaration
     *
     *
     * @param degenerated
     *
     * @throws IOException
     *
     * @see
     */
    public void checkPending(boolean degenerated) throws IOException {
	if (pending == null) {
	    return;
	} 

	int len = attributes.size();

	buf.write(len == 0 ? (degenerated ? Wbxml.LITERAL : Wbxml.LITERAL_C) 
		  : (degenerated ? Wbxml.LITERAL_A : Wbxml.LITERAL_AC));
	writeStrT(pending);

	for (int i = 0; i < len; ) {
	    buf.write(Wbxml.LITERAL);
	    writeStrT((String) attributes.elementAt(i++));
	    buf.write(Wbxml.STR_I);
	    writeStrI(buf, (String) attributes.elementAt(i++));
	} 

	pending = null;

	attributes.removeAllElements();
    } 

    /**
     * Method declaration
     *
     *
     * @param prefixMap
     * @param name
     *
     * @throws IOException
     *
     * @see
     */
    public void writeStartTag(PrefixMap prefixMap, 
			      String name) throws IOException {
	current = new State(current, prefixMap, name);

	checkPending(false);

	pending = name;
    } 

    /**
     * Method declaration
     *
     *
     * @param name
     * @param value
     *
     * @see
     */
    public void writeAttribute(String name, String value) {
	attributes.addElement(name);
	attributes.addElement(value);
    } 

    /**
     * Method declaration
     *
     *
     * @param chars
     * @param start
     * @param len
     *
     * @throws IOException
     *
     * @see
     */
    public void write(char[] chars, int start, int len) throws IOException {
	checkPending(false);
	buf.write(Wbxml.STR_I);
	writeStrI(buf, new String(chars, start, len));
    } 

    /**
     * Method declaration
     *
     *
     * @throws IOException
     *
     * @see
     */
    public void writeEndTag() throws IOException {
	current = current.prev;

	if (pending != null) {
	    checkPending(true);
	} else {
	    buf.write(Wbxml.END);
	}
    } 

    /**
     * currently ignored!
     */
    public void writeLegacy(int type, String data) {}

    // ------------- internal methods --------------------------

    /**
     * Method declaration
     *
     *
     * @param out
     * @param i
     *
     * @throws IOException
     *
     * @see
     */
    static void writeInt(OutputStream out, int i) throws IOException {
	byte[] buf = new byte[5];
	int    idx = 0;

	do {
	    buf[idx++] = (byte) (i & 0x7f);
	    i = i >> 7;
	} while (i != 0);

	while (idx > 1) {
	    out.write(buf[--idx] | 0x80);
	} 

	out.write(buf[0]);
    } 

    /**
     * Method declaration
     *
     *
     * @param out
     * @param s
     *
     * @throws IOException
     *
     * @see
     */
    static void writeStrI(OutputStream out, String s) throws IOException {
	for (int i = 0; i < s.length(); i++) {
	    out.write((byte) s.charAt(i));
	} 

	out.write(0);
    } 

    /**
     * Method declaration
     *
     *
     * @param s
     *
     * @throws IOException
     *
     * @see
     */
    void writeStrT(String s) throws IOException {
	Integer idx = (Integer) stringTable.get(s);

	if (idx == null) {
	    idx = new Integer(stringTableBuf.size());

	    stringTable.put(s, idx);
	    writeStrI(stringTableBuf, s);
	    stringTableBuf.flush();
	} 

	writeInt(buf, idx.intValue());
    } 

}

